import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        rubbish_pos = positions['rubbish']  # Position of the rubbish
        bin_pos = positions['bin']           # Position of the bin
        bottom_anchor_pos = positions['bottom_anchor_pos']  # Focus on bottom drawer
        side_pos_bottom = positions['bottom_side_pos'] # Position for side of bottom drawer

        # === Task Execution Steps ===
        
        # Step 1: Rotate the gripper for drawer orientation
        print("[Task] Rotating gripper to 90 degrees for side position...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0])) 
        if done:
            print("[Task] Gripper rotated successfully.")
        else:
            print("[Task] Failed to rotate gripper.")

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to side position of the bottom drawer...")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom) 
        if done:
            print("[Task] Moved to side position successfully.")
        else:
            print("[Task] Failed to move to side position.")

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to anchor position of the bottom drawer...")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos) 
        if done:
            print("[Task] Moved to anchor position successfully.")
        else:
            print("[Task] Failed to move to anchor position.")

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer to open it...")
        obs, reward, done = pick_drawer(env, task, g='gripper', d='bottom', p='anchor_pos_bottom') 
        if done:
            print("[Task] Successfully picked the bottom drawer.")
        else:
            print("[Task] Failed to pick the bottom drawer.")

        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer to open it...")
        obs, reward, done = pull(env, task, pull_distance=0.1)  
        if done:
            print("[Task] Successfully opened the bottom drawer.")
        else:
            print("[Task] Failed to open the bottom drawer.")

        # Step 6: Pick up the rubbish from the table
        print("[Task] Picking up the rubbish from the table...")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)  
        if done:
            print("[Task] Successfully picked up the rubbish.")
        else:
            print("[Task] Failed to pick up the rubbish.")

        # Step 7: Place the rubbish into the bin
        print("[Task] Placing the rubbish into the bin...")
        obs, reward, done = place(env, task, target_pos=bin_pos) 
        if done:
            print("[Task] Successfully placed the rubbish in the bin.")
        else:
            print("[Task] Failed to place the rubbish in the bin.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()