import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bin_pos = positions['bin']
        rubbish_pos = positions['rubbish']

        # Step 1: Rotate to zero degrees to face the drawer
        print("[Task] Rotating gripper to face the drawer.")
        obs, reward, done = rotate(env, task, target_quat=normalize_quaternion([0, 0, 0, 1]), max_steps=100, threshold=0.05, timeout=10.0)
        
        if done:
            print("[Task] Ended after rotating.")
        
        # Step 2: Move to the side of the drawer
        print("[Task] Moving to the side of the drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Ended after moving to the side position.")
        
        # Step 3: Move to anchor position of the drawer
        print("[Task] Moving to anchor position of the drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Ended after moving to anchor position.")
        
        # Step 4: Pick the drawer
        print("[Task] Picking up the drawer.")
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Ended after picking the drawer.")
        
        # Step 5: Pull open the drawer
        print("[Task] Pulling open the drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x', max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Ended after pulling open the drawer.")
        
        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Ended after picking the rubbish.")
        
        # Step 7: Place the rubbish into the bin
        print("[Task] Placing rubbish into the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()