import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # Step 1: Rotate the gripper to ninety degrees
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Assuming rotation to be represented this way
        if done:
            print("[Task] Rotation successful!")
        else:
            print("[Task] Failed to rotate the gripper.")
            return

        # Step 2: Move to the side position near the drawer
        print("[Task] Moving to the side position near the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved successfully!")
        else:
            print("[Task] Failed to move to the side position.")
            return
        
        # Step 3: Move to the anchor position of the drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to the anchor position successfully!")
        else:
            print("[Task] Failed to move to the anchor position.")
            return
        
        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'], approach_distance=0.1)
        if done:
            print("[Task] Drawer picked successfully!")
        else:
            print("[Task] Failed to pick the drawer.")
            return
        
        # Step 5: Pull the drawer to unlock the cabinet
        print("[Task] Pulling the drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        if done:
            print("[Task] Drawer pulled successfully!")
        else:
            print("[Task] Failed to pull the drawer.")
            return

        # Step 6: Pick the rubbish from the table
        print("[Task] Picking up the rubbish from the table.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.1)
        if done:
            print("[Task] Rubbish picked successfully!")
        else:
            print("[Task] Failed to pick the rubbish.")
            return

        # Step 7: Place the rubbish into the bin
        print("[Task] Placing rubbish into the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.1)
        if done:
            print("[Task] Rubbish placed into the bin successfully!")
        else:
            print("[Task] Failed to place the rubbish into the bin.")
  
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()