import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define positions of drawer and plate
        bottom_drawer_pos = positions['bottom_anchor_pos']
        plate_pos = positions['plate']

        # === Oracle Plan Execution ===
        # Step 1: Rotate the gripper to a proper position (e.g., zero degrees)
        print("[Task] Rotating gripper.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05)
        if done:
            print("[Task] Gripper rotated successfully.")
        else:
            print("[Task] Failed to rotate the gripper.")
            return

        # Step 2: Move the gripper to the side position of the bottom drawer
        print("[Task] Moving to side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'], max_steps=100, threshold=0.01)
        if done:
            print("[Task] Moved successfully to the side of the bottom drawer.")
        else:
            print("[Task] Failed to move to the side position.")
            return
            
        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'], max_steps=100, threshold=0.01)
        if done:
            print("[Task] Moved successfully to the anchor of the bottom drawer.")
        else:
            print("[Task] Failed to move to the anchor position.")
            return

        # Step 4: Pick the drawer
        print("[Task] Picking the drawer.")
        obs, reward, done = pick_drawer(env, task, 'gripper', 'bottom', positions['bottom_anchor_pos'])
        if done:
            print("[Task] Drawer picked successfully.")
        else:
            print("[Task] Failed to pick the drawer.")
            return
            
        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.0, pull_axis='z', max_steps=100, threshold=0.01)
        if done:
            print("[Task] Drawer opened successfully.")
        else:
            print("[Task] Failed to open the drawer.")
            return

        # Step 6: Pick tomato1 from the table
        print("[Task] Picking tomato1 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'], approach_distance=0.15)
        if done:
            print("[Task] Picked tomato1 successfully.")
        else:
            print("[Task] Failed to pick tomato1.")
            return

        # Step 7: Place tomato1 on the plate
        print("[Task] Placing tomato1 on the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])
        if done:
            print("[Task] Placed tomato1 successfully.")
        else:
            print("[Task] Failed to place tomato1.")
            return

        # Step 8: Pick tomato2 from the table
        print("[Task] Picking tomato2 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'], approach_distance=0.15)
        if done:
            print("[Task] Picked tomato2 successfully.")
        else:
            print("[Task] Failed to pick tomato2.")
            return

        # Step 9: Place tomato2 on the plate
        print("[Task] Placing tomato2 on the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])
        if done:
            print("[Task] Placed tomato2 successfully.")
        else:
            print("[Task] Failed to place tomato2.")
            return

        print("[Task] All tasks completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()