import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Initialize required parameters based on the provided plan
        gripper = 'gripper'
        tomato1 = 'tomato1'
        tomato2 = 'tomato2'
        plate = 'plate'
        bottom_drawer = 'bottom'
        side_pos_bottom = positions['bottom_side_pos']
        anchor_pos_bottom = positions['bottom_anchor_pos']

        # Step 1: Rotate gripper to 90 degrees
        print("[Task] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, np.array([0, 0, 1, 0]))  # Quaternion for 90 degrees
        if done: return

        # Step 2: Move gripper to side position of the bottom drawer
        print("[Task] Moving to side position of the bottom drawer.")
        obs, reward, done = move(env, task, side_pos_bottom)
        if done: return

        # Step 3: Move gripper to anchor position of the bottom drawer
        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, anchor_pos_bottom)
        if done: return

        # Step 4: Pick up the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, gripper, bottom_drawer, anchor_pos_bottom)
        if done: return

        # Step 5: Pull to open the bottom drawer
        print("[Task] Pulling the bottom drawer to open it.")
        obs, reward, done = pull(env, task, 0.2, pull_axis='x')  # Assumed pull distance and axis
        if done: return

        # Step 6: Pick up the first tomato (tomato1)
        print("[Task] Picking up the first tomato.")
        obs, reward, done = pick(env, task, tomato1, positions['table'])  # Assuming 'table' is the correct position
        if done: return

        # Step 7: Place the first tomato on the plate
        print("[Task] Placing the first tomato on the plate.")
        obs, reward, done = place(env, task, tomato1, plate)
        if done: return

        # Step 8: Pick up the second tomato (tomato2)
        print("[Task] Picking up the second tomato.")
        obs, reward, done = pick(env, task, tomato2, positions['table'])  # Assuming 'table' is the location
        if done: return

        # Step 9: Place the second tomato on the plate
        print("[Task] Placing the second tomato on the plate.")
        obs, reward, done = place(env, task, tomato2, plate)
        if done: return
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()