import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define key object positions based on the retrieved data
        bottom_pos = positions['bottom_anchor_pos']
        side_pos_bottom = positions['bottom_side_pos']
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']

        # === Execute the Oracle Plan ===

        # Step 1: Rotate the gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100)
        if done:
            print("[Task] Rotated gripper successfully.")
        else:
            print("[Task] Failed to rotate gripper.")
            return

        # Step 2: Move to the side position for the bottom drawer
        obs, reward, done = move(env, task, target_pos=side_pos_bottom, max_steps=100)
        if done:
            print("[Task] Moved to side position successfully.")
        else:
            print("[Task] Failed to move to side position.")
            return

        # Step 3: Move to the anchor position for the bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_pos, max_steps=100)
        if done:
            print("[Task] Moved to anchor position successfully.")
        else:
            print("[Task] Failed to move to anchor position.")
            return

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick_drawer(env, task, target_pos=bottom_pos)
        if done:
            print("[Task] Picked the bottom drawer successfully.")
        else:
            print("[Task] Failed to pick the bottom drawer.")
            return

        # Step 5: Pull the bottom drawer to unlock the cabinet
        obs, reward, done = pull(env, task, pull_distance=1.0)
        if done:
            print("[Task] Pulled the bottom drawer successfully.")
        else:
            print("[Task] Failed to pull the bottom drawer.")
            return

        # Step 6: Pick the first tomato from the table
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Picked the first tomato successfully.")
        else:
            print("[Task] Failed to pick the first tomato.")
            return

        # Step 7: Place the first tomato on the plate
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Placed the first tomato on the plate successfully.")
        else:
            print("[Task] Failed to place the first tomato on the plate.")
            return

        # Step 8: Pick the second tomato from the table
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Picked the second tomato successfully.")
        else:
            print("[Task] Failed to pick the second tomato.")
            return

        # Step 9: Place the second tomato on the plate
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Placed the second tomato on the plate successfully.")
        else:
            print("[Task] Failed to place the second tomato on the plate.")
            return

        print("[Task] All tasks completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()