import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        bottom_pos = positions['bottom_anchor_pos']
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']

        # === Execute the Plan ===
        print("[Task] Rotating gripper to 90 degrees...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))
        
        if done:
            print("[Task] Rotation completed!")
        else:
            print("[Task] Rotation not completed.")

        print("[Task] Moving to the side of the bottom drawer...")
        obs, reward, done = move(env, task, target_pos=bottom_pos)
        
        if done:
            print("[Task] Moved to the side position of the bottom drawer!")
        else:
            print("[Task] Move action not completed.")

        print("[Task] Moving to the anchor position of the bottom drawer...")
        obs, reward, done = move_to_anchor(env, task, target_pos=bottom_pos)
        
        if done:
            print("[Task] Moved to anchor position of bottom drawer!")
        else:
            print("[Task] Move action to anchor position not completed.")

        print("[Task] Picking the bottom drawer...")
        obs, reward, done = pick_drawer(env, task, gripper='gripper', drawer='bottom', position='anchor-pos-bottom')
        
        if done:
            print("[Task] Bottom drawer picked successfully!")
        else:
            print("[Task] Failed to pick the bottom drawer.")

        print("[Task] Pulling open the bottom drawer...")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        
        if done:
            print("[Task] Pull action completed!")
        else:
            print("[Task] Pull action not completed.")

        print("[Task] Picking up tomato 1...")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Tomato 1 picked successfully!")
        else:
            print("[Task] Failed to pick Tomato 1.")

        print("[Task] Placing tomato 1 on the plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        
        if done:
            print("[Task] Tomato 1 placed successfully!")
        else:
            print("[Task] Failed to place Tomato 1.")

        print("[Task] Picking up tomato 2...")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Tomato 2 picked successfully!")
        else:
            print("[Task] Failed to pick Tomato 2.")

        print("[Task] Placing tomato 2 on the plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        
        if done:
            print("[Task] Tomato 2 placed successfully!")
        else:
            print("[Task] Failed to place Tomato 2.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()