import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define positions based on the task
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        middle_side_pos = positions['middle_side_pos']
        middle_anchor_pos = positions['middle_anchor_pos']

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper
        print("[Task] Rotating gripper to ninety degrees.")
        rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))

        # Step 2: Move the gripper to the side position of the middle drawer
        print("[Task] Moving to the side position of the middle drawer.")
        move(env, task, middle_side_pos)

        # Step 3: Move to the anchor position of the middle drawer
        print("[Task] Moving to anchor position of the middle drawer.")
        move(env, task, middle_anchor_pos)

        # Step 4: Pick the middle drawer
        print("[Task] Picking the middle drawer.")
        pick_drawer(env, task, 'middle', middle_anchor_pos)

        # Step 5: Pull the drawer to open it
        print("[Task] Pulling the middle drawer to open.")
        pull(env, task, pull_distance=0.1)

        # Step 6: Pick the first tomato
        print("[Task] Picking tomato1.")
        pick(env, task, target_pos=tomato1_pos)

        # Step 7: Place the first tomato on the plate
        print("[Task] Placing tomato1 on the plate.")
        place(env, task, target_pos=plate_pos)

        # Step 8: Pick the second tomato
        print("[Task] Picking tomato2.")
        pick(env, task, target_pos=tomato2_pos)

        # Step 9: Place the second tomato on the plate
        print("[Task] Placing tomato2 on the plate.")
        place(env, task, target_pos=plate_pos)

        print("[Task] All tomatoes placed successfully!")
    
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()