import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define actual positions based on the object list
        side_pos_bottom = positions['bottom_side_pos']  # Position for side of the bottom drawer
        anchor_pos_bottom = positions['bottom_anchor_pos']  # Position for anchor of the bottom drawer
        plate_pos = positions['plate']  # Position for the plate
        tomato1_pos = positions['tomato1']  # Position for tomato1
        tomato2_pos = positions['tomato2']  # Position for tomato2

        # === Execute Plan Steps ===
        
        # Step 1: Rotate to prepare for side movement
        print("[Task] Rotating gripper to prepare for side movement.")
        obs, reward, done = rotate(env, task, target_quat=[0, 0, 1, 0])  # Rotate gripper
        if done:
            print("[Task] Task ended after rotation!")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)  # Move to side position
        if done:
            print("[Task] Task ended after moving to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)  # Move to anchor position
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=anchor_pos_bottom)  # Pick the drawer
        if done:
            print("[Task] Task ended after picking drawer!")
            return
            
        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.5, pull_axis='x')  # Specify pull parameters
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6: Pick up tomato 1
        print("[Task] Picking up tomato1.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)  # Pick tomato1
        if done:
            print("[Task] Task ended after picking tomato1!")
            return
        
        # Step 7: Place tomato 1 on the plate
        print("[Task] Placing tomato1 on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)  # Place tomato1 on plate
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: Pick up tomato 2
        print("[Task] Picking up tomato2.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)  # Pick tomato2
        if done:
            print("[Task] Task ended after picking tomato2!")
            return
        
        # Step 9: Place tomato 2 on the plate
        print("[Task] Placing tomato2 on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)  # Place tomato2 on plate
        if done:
            print("[Task] Task ended after placing tomato2!")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()