def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        drawer_position = positions['drawer']  # Assuming this is the position of the drawer
        tomato_position = positions['tomato']  # Position of the tomato
        plate_position = positions['plate']      # Position of the plate
        gripper_position = positions['gripper']  # Current gripper position
        
        # Get known objects from the task's observation
        is_drawer_open = obs.is_drawer_open
        is_drawer_locked = obs.is_locked
        
        # Plan Execution
        # Step 1: Move to the drawer
        print("[Task] Moving to drawer at:", drawer_position)
        obs, reward, done = move(env, gripper_position, drawer_position)
        if done: 
            print("[Task] Task ended while moving to drawer!")
            return
        
        # Step 2: Check if the drawer is locked, if so, unlock it
        if is_drawer_locked:
            print("[Task] Drawer is locked, need to unlock.")
            obs, reward, done = pull(env, gripper_position, 'drawer')  # Simulated action to pull handle and unlock
            if done: 
                print("[Task] Task ended while unlocking drawer!")
                return
        
        # Step 3: Open the drawer if it is not already opened
        if not is_drawer_open:
            print("[Task] Opening the drawer.")
            obs, reward, done = pull(env, gripper_position, 'drawer')  # Action to open the drawer
            if done: 
                print("[Task] Task ended while opening drawer!")
                return
        
        # Step 4: Move to the tomato
        print("[Task] Moving to tomato at:", tomato_position)
        obs, reward, done = move(env, drawer_position, tomato_position)
        if done: 
            print("[Task] Task ended while moving to tomato!")
            return
        
        # Step 5: Pick the tomato
        print("[Task] Picking the tomato.")
        obs, reward, done = pick(env, 'tomato', tomato_position, gripper_position)
        if done:
            print("[Task] Finished picking the tomato!")
            return
        
        # Step 6: Move to the plate
        print("[Task] Moving to plate at:", plate_position)
        obs, reward, done = move(env, tomato_position, plate_position)
        if done: 
            print("[Task] Task ended while moving to plate!")
            return
        
        # Step 7: Place the tomato on the plate
        print("[Task] Placing the tomato on the plate.")
        obs, reward, done = place(env, 'tomato', plate_position)
        if done: 
            print("[Task] Finished placing the tomato on the plate!")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()