import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        tomato1_key = 'tomato1' if 'tomato1' in positions else 'item1'
        tomato2_key = 'tomato2'
        plate_key = 'plate'

        if tomato1_key not in positions or tomato2_key not in positions or plate_key not in positions:
            raise KeyError(f"Missing required object positions: {tomato1_key}, {tomato2_key}, or {plate_key}")

        tomato1_pos = positions[tomato1_key]
        tomato2_pos = positions[tomato2_key]
        plate_pos = positions[plate_key]

        # === Oracle Plan Execution ===

        # Step 1: Pick the first tomato
        print(f"[Plan] Step 1: pick {tomato1_key} from table at {tomato1_pos}")
        obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        if done:
            print("[Task] Terminated early after first place. Reward:", reward)
            return

        # Step 3: Pick the second tomato
        print(f"[Plan] Step 3: pick {tomato2_key} from table at {tomato2_pos}")
        obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        if done:
            print("[Task] Terminated early after second pick. Reward:", reward)
            return

        # Step 4: Place the second tomato onto the plate
        print(f"[Plan] Step 4: place {tomato2_key} onto plate at {plate_pos}")
        obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        if done:
            print("[Task] Completed after second place! Final Reward:", reward)
        else:
            print("[Task] Finished plan execution. Done flag:", done, "Final Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()