# run_skeleton_task.py
#
# Description:
#  - This code is a skeleton that shows how to set up an environment, retrieve
#    object positions, and execute a plan without being tied to a specific skill
#    like pick or place.
#  - Use this as a base for creating your own task logic.
#  - The sections that are task-dependent (e.g., manipulation steps) are marked
#    with TODO or placeholders.

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # you don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Validate required positions
        if 'switch' not in positions:
            raise KeyError("switch position not found in environment")
        if 'plate' not in positions:
            raise KeyError("plate position not found in environment")
        if 'tomato1' not in positions or 'tomato2' not in positions:
            raise KeyError("required tomatoes not found in environment")
        switch_pos = positions['switch']
        plate_pos = positions['plate']

        # === Task Plan: Put all tomatoes on the plate ===
        # 1) Close gripper so we can press the switch
        print("[Task] Closing gripper to prepare for pressing switch.")
        [Frozen Code Start]
        obs, reward, done = close_gripper(env, task)
        obs, reward, done = press(env, task, target_pos=switch_pos)
        [Frozen Code End]

        if done:
            print("[Task] Task ended after pressing switch! Reward:", reward)
            return

        # Steps from specification:
        # Step 1: pick tomato1 from table
        print("[Task] Picking tomato1 at:", positions['tomato1'])
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'])
        if done:
            print("[Task] Task ended after picking tomato1! Reward:", reward)
            return

        # Step 2: place tomato1 on plate
        print("[Task] Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato1! Reward:", reward)
            return

        # Step 3: pick tomato2 from table
        print("[Task] Picking tomato2 at:", positions['tomato2'])
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'])
        if done:
            print("[Task] Task ended after picking tomato2! Reward:", reward)
            return

        # Step 4: place tomato2 on plate
        print("[Task] Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato2! Reward:", reward)
            return

        # All steps executed
        print("[Task] All tomatoes placed on plate. Task completed! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()