import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        tomato1_pos = np.array(positions['tomato1'])
        tomato2_pos = np.array(positions['tomato2'])
        plate_pos   = np.array(positions['plate'])
        switch_pos  = np.array(positions['switch'])

        # === Oracle Plan Execution ===

        # Step 1: close gripper
        print("[Task] Step 1: Closing gripper")
        try:
            obs, reward, done = close_gripper(env, task)
        except Exception as e:
            print(f"[Error] close_gripper failed: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after closing gripper! Reward:", reward)
            return

        # Step 2: press switch
        print("[Task] Step 2: Pressing switch at:", switch_pos)
        try:
            obs, reward, done = press(
                env,
                task,
                target_pos=switch_pos,
                max_steps=100,
                threshold=0.005,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] press(switch) failed: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after pressing switch! Reward:", reward)
            return

        # Step 3-6: pick and place tomatoes (frozen code)
        obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

        # Final status
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task completed with done=False. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()