import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Identify trash objects: everything except bin, plate, switch
        non_trash = ('bin', 'plate', 'switch')
        trash_objects = [name for name in positions.keys() if name not in non_trash]

        # === Task Plan Execution ===
        for obj_name in trash_objects:
            obj_pos = positions.get(obj_name)
            if obj_pos is None:
                print(f"[Error] Position for object '{obj_name}' not found, skipping.")
                continue

            # Step 1: Pick the trash object
            print(f"[Task] Picking up '{obj_name}' at position: {obj_pos}")
            try:
                obs, reward, done = pick(
                    env=env,
                    task=task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Exception during pick of '{obj_name}': {e}")
                break
            if done:
                print(f"[Task] Task ended prematurely during pick of '{obj_name}'.")
                break

            # Step 2: Place the object into the bin
            bin_pos = positions.get('bin')
            if bin_pos is None:
                print("[Error] Bin position not found. Aborting.")
                break
            print(f"[Task] Placing '{obj_name}' into bin at position: {bin_pos}")
            try:
                obs, reward, done = place(
                    env=env,
                    task=task,
                    target_pos=np.array(bin_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Exception during place of '{obj_name}': {e}")
                break
            if done:
                print(f"[Task] Task ended prematurely during place of '{obj_name}'.")
                break

        else:
            # Executed if loop completes without break
            print("[Task] All trash objects have been thrown away successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
