import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper (no-op rotation to current orientation)
        print("[Task] Step 1: Rotate gripper to prepare for drawer interaction")
        obs = task.get_observation()
        try:
            curr_quat = normalize_quaternion(obs.gripper_pose[3:7])
            obs, reward, done = rotate(env, task, target_quat=curr_quat)
        except Exception as e:
            print(f"[Task][Error] Rotation failed: {e}")
            raise
        if done:
            print("[Task] Ended after rotation.")
            return

        # Step 2: Move gripper to drawer handle (bottom_side_pos)
        print("[Task] Step 2: Move to drawer handle position")
        try:
            handle_pos = np.array(positions['bottom_side_pos'])
            obs, reward, done = move(env, task, target_pos=handle_pos)
        except KeyError:
            raise RuntimeError("Drawer handle position 'bottom_side_pos' not found")
        except Exception as e:
            print(f"[Task][Error] Move to handle failed: {e}")
            raise
        if done:
            print("[Task] Ended after moving to handle.")
            return

        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat=curr_quat)
        obs, reward, done = move(env, task, target_pos=handle_pos)
        # Frozen Code End

        # Step 3: Pull to open the drawer
        print("[Task] Step 3: Pull to open the drawer")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        except Exception as e:
            print(f"[Task][Error] Pull failed: {e}")
            raise
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 4: Pick up the rubbish
        print("[Task] Step 4: Pick up the rubbish")
        try:
            rubbish_pos = np.array(positions['rubbish'])
            obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        except KeyError:
            raise RuntimeError("Object 'rubbish' not found in positions")
        except Exception as e:
            print(f"[Task][Error] Pick failed: {e}")
            raise
        if done:
            print("[Task] Ended after picking rubbish.")
            return

        # Step 5: Place the rubbish into the bin
        print("[Task] Step 5: Place the rubbish into the bin")
        try:
            bin_pos = np.array(positions['bin'])
            obs, reward, done = place(env, task, target_pos=bin_pos)
        except KeyError:
            raise RuntimeError("Object 'bin' not found in positions")
        except Exception as e:
            print(f"[Task][Error] Place failed: {e}")
            raise
        if done:
            print(f"[Task] Task completed successfully! Final reward: {reward}")
        else:
            print("[Task] Task finished all steps but done=False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()