[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: open bottom drawer fully, pick up rubbish, place in bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()

        # Initialize video recording
        init_video_writers(obs)

        # Wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Task] Retrieved object positions:", positions.keys())

        # === Define Orientations ===
        # zero_deg: no rotation around z; ninety_deg: 90° around z
        zero_quat = R.from_euler('z', 0, degrees=True).as_quat()
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to ninety_deg
        print("[Task] Step 1: Rotating gripper to 90° around Z")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Early exit after rotation. Reward:", reward)
            return

        # Step 2: Move gripper to side position of bottom drawer
        bottom_side = positions.get('bottom_side_pos')
        print("[Task] Step 2: Moving gripper to bottom side position", bottom_side)
        obs, reward, done = move(env, task, np.array(bottom_side))
        if done:
            print("[Task] Early exit after move to side. Reward:", reward)
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        bottom_anchor = positions.get('bottom_anchor_pos')
        print("[Task] Step 3: Moving gripper to bottom anchor position", bottom_anchor)
        obs, reward, done = move(env, task, np.array(bottom_anchor))
        if done:
            print("[Task] Early exit after move to anchor. Reward:", reward)
            return

        # Step 4: Pull the bottom drawer fully open
        bottom_joint = positions.get('bottom_joint_pos')
        # Compute pull distance along X
        pull_distance = np.linalg.norm(np.array(bottom_anchor) - np.array(bottom_joint))
        print(f"[Task] Step 4: Pulling drawer by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Early exit after pull. Reward:", reward)
            return

        # Step 5: Pick up the rubbish from table
        rubbish_pos = positions.get('rubbish')
        print("[Task] Step 5: Picking up rubbish at", rubbish_pos)
        obs, reward, done = pick(env, task, np.array(rubbish_pos))
        if done:
            print("[Task] Early exit after picking rubbish. Reward:", reward)
            return

        # Step 6: Place the rubbish in the bin
        bin_pos = positions.get('bin')
        print("[Task] Step 6: Placing rubbish in bin at", bin_pos)
        obs, reward, done = place(env, task, np.array(bin_pos))
        if done:
            print("[Task] Completed placing rubbish. Reward:", reward)
        else:
            print("[Task] Task ended with done=False. Reward:", reward)

    except Exception as e:
        print("[Task] Exception occurred during execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()