from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Generic skeleton for running the "open drawer then bin rubbish" task.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all relevant positions and quaternions
        positions = get_object_positions()
        zero_quat = positions['zero_deg']
        ninety_quat = positions['ninety_deg']
        side_bottom_pos = positions['bottom_side_pos']
        anchor_bottom_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Rotating gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Ended during rotate.")
            return

        # Step 2: Move to side position of bottom drawer
        print("[Task] Moving to side position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=np.array(side_bottom_pos))
        if done:
            print("[Task] Ended during move-to-side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Task] Moving to anchor position of bottom drawer")
        obs, reward, done = move(env, task, target_pos=np.array(anchor_bottom_pos))
        if done:
            print("[Task] Ended during move-to-anchor.")
            return

        # Step 4: Pull drawer open along the x axis by 0.1m
        print("[Task] Pulling the bottom drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended during pull.")
            return

        # Step 5: Pick up the rubbish from the table
        print("[Task] Picking up the rubbish")
        obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos))
        if done:
            print("[Task] Ended during pick.")
            return

        # Step 6: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin")
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
        if done:
            print("[Task] Completed successfully! Final reward:", reward)
        else:
            print("[Task] Finished all steps but 'done' flag is False. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()