import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Initialization ===
        # Compute quaternion for 90° rotation around Z
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        # Define pull distance for opening the drawer
        pull_distance = 0.1

        # === Oracle Plan Execution ===

        # 1) Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended prematurely after move to side-pos-bottom.")
            return

        # 2) Rotate the gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Task ended prematurely after rotate to ninety_deg.")
            return

        # 3) Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended prematurely after move to anchor-pos-bottom.")
            return

        # 4) Pull the drawer open
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended prematurely after pull.")
            return

        # 5) Pick up any rubbish on the table
        obs, reward, done = pick(env, task, target_pos=positions['rubbish'])
        if done:
            print("[Task] Task ended prematurely after pick rubbish.")
            return

        # 6) Move to the bin
        obs, reward, done = move(env, task, target_pos=positions['bin'])
        if done:
            print("[Task] Task ended prematurely after move to bin.")
            return

        # 7) Place the rubbish into the bin
        print("[Task] Placing rubbish into bin at:", positions['bin'])
        obs, reward, done = place(env, task, target_pos=positions['bin'])
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Place action executed, but done=False. Current reward:", reward)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()