import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Key object positions we need for the plan
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # === Plan Execution ===

        # 1) Move to the side of the bottom drawer
        print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended unexpectedly during move to drawer side.")
            return

        # 2) Rotate the gripper (no-op to satisfy orientation requirement)
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])
        print("[Task] Rotating gripper to maintain orientation (no-op):", current_quat)
        obs, reward, done = rotate(env, task, target_quat=current_quat)
        if done:
            print("[Task] Task ended unexpectedly during rotate.")
            return

        # 3) Pull the bottom drawer open
        pull_distance = np.linalg.norm(bottom_anchor_pos - bottom_side_pos)
        print(f"[Task] Pulling bottom drawer by distance: {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended unexpectedly during pull.")
            return

        # 4) Pick up the rubbish
        print("[Task] Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, rubble_pos if False else rubbish_pos)
        # Note: using the provided rubbish_pos variable
        if done:
            print("[Task] Task ended unexpectedly during pick.")
            return

        # 5) Place the rubbish into the bin
        print("[Task] Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task] Task ended during place.")
            return

        # Final status
        if reward is not None:
            print(f"[Task] Completed plan. Final reward: {reward}")
        else:
            print("[Task] Plan executed. done flag:", done)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()