import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Move gripper to the side position of the bottom drawer
        print("[Task] Step 1: move to side-pos-bottom")
        side_bottom_pos = np.array(positions['side-pos-bottom'])
        obs, reward, done = move(env, task, side_bottom_pos)
        if done:
            print("[Task] Ended early during move. Reward:", reward)
            return

        # Step 2: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 2: rotate to ninety_deg")
        ninety_quat = np.array(positions['ninety_deg'])
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Ended early during rotate. Reward:", reward)
            return

        # Step 3: Pull the drawer open along the x-axis
        print("[Task] Step 3: pull drawer")
        pull_distance = 0.1  # adjust as needed for your drawer
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Ended early during pull. Reward:", reward)
            return

        # Step 4: Pick tomato1 from the table
        print("[Task] Step 4: pick tomato1")
        tomato1_pos = np.array(positions['tomato1'])
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Ended early during pick(tomato1). Reward:", reward)
            return

        # Step 5: Place tomato1 on the plate
        print("[Task] Step 5: place tomato1 on plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended early during place(tomato1). Reward:", reward)
            return

        # === Frozen Code Start ===
        obs, reward, done = move(env, task, side_bottom_pos)
        obs, reward, done = rotate(env, task, ninety_quat)
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        obs, reward, done = pick(env, task, tomato1_pos)
        obs, reward, done = place(env, task, plate_pos)
        # === Frozen Code End ===

        # Step 6: Pick tomato2 from the table
        print("[Task] Step 6: pick tomato2")
        tomato2_pos = np.array(positions['tomato2'])
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Ended early during pick(tomato2). Reward:", reward)
            return

        # Step 7: Place tomato2 on the plate
        print("[Task] Step 7: place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended early during place(tomato2). Reward:", reward)
            return

        print("[Task] Plan execution completed successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()