import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Choose the bottom drawer
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']
        joint_pos = positions['bottom_joint_pos']
        # Items and plate
        item1_pos = positions['item1']
        item2_pos = positions['item2']
        plate_pos = positions['plate']

        # Compute required parameters
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()
        pull_distance = np.linalg.norm(joint_pos - anchor_pos)

        # === Execute Oracle Plan ===
        # Step 1–8: move, rotate, move, pull, pick, place, pick, place
        [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        [Frozen Code End]

        # === Final Status ===
        if done:
            print(f"[Task] Completed successfully with reward {reward}")
        else:
            print(f"[Task] Plan executed; done flag = {done}, last reward = {reward}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()