# run_skeleton_task.py
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running the combined-domain task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if recording is desired
        init_video_writers(obs)

        # Wrap the task.step and task.get_observation to record video frames
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Pre-compute Key Poses and Orientations ===
        print("[Task] Rotating gripper to 90 degrees about Z axis")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()

        print("[Task] Retrieving bottom side and anchor positions")
        side_pos = np.array(positions['bottom_side_pos'])
        anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Execute Oracle Plan ===

        # Step 1-4 are frozen: rotate, move to side, move to anchor, pull drawer
        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        [Frozen Code End]

        if done:
            print("[Task] Early termination after drawer pull. Reward:", reward)
            return

        # Step 5: Pick the first tomato (tomato1)
        print("[Task] Picking up tomato1")
        tom1_pos = np.array(positions['tomato1'])
        obs, reward, done = pick(env, task, tom1_pos)
        if done:
            print("[Task] Early termination after picking tomato1. Reward:", reward)
            return

        # Step 6: Place the first tomato onto the plate
        print("[Task] Placing tomato1 on the plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Early termination after placing tomato1. Reward:", reward)
            return

        # Step 7: Pick the second tomato (tomato2)
        print("[Task] Picking up tomato2")
        tom2_pos = np.array(positions['tomato2'])
        obs, reward, done = pick(env, task, tom2_pos)
        if done:
            print("[Task] Early termination after picking tomato2. Reward:", reward)
            return

        # Step 8: Place the second tomato onto the plate
        print("[Task] Placing tomato2 on the plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Plan executed; done flag is False. Current reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()