import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan: Open bottom drawer, then move all tomatoes to plate ===
        # 1) Rotate gripper to 90 degrees (around Z)
        print("[Task] Rotating gripper by 90 degrees")
        # Quaternion for 90° about Z axis in xyzw: [0,0,sin(pi/4),cos(pi/4)]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        # 2) Move to side position of bottom drawer
        print("[Task] Moving to bottom side position")
        side_pos = positions.get('bottom_side_pos')
        if side_pos is None:
            raise KeyError("bottom_side_pos not found in positions")
        # 3) Move to anchor position of bottom drawer
        print("[Task] Moving to bottom anchor position (drawer handle)")
        anchor_pos = positions.get('bottom_anchor_pos')
        if anchor_pos is None:
            raise KeyError("bottom_anchor_pos not found in positions")
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        # 4) Pick the drawer handle (close gripper around it)
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.05, approach_axis='x')
        if done:
            print("[Task] Early termination after pick handle. Reward:", reward)
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Early termination after pull. Reward:", reward)
            return

        # 6) Pick tomato1 from table
        print("[Task] Picking tomato1")
        tomato1_pos = positions.get('tomato1')
        if tomato1_pos is None:
            raise KeyError("tomato1 not found in positions")
        obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Early termination after pick tomato1. Reward:", reward)
            return

        # 7) Place tomato1 on plate
        print("[Task] Placing tomato1 on plate")
        plate_pos = positions.get('plate')
        if plate_pos is None:
            raise KeyError("plate not found in positions")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Early termination after place tomato1. Reward:", reward)
            return

        # 8) Pick tomato2 from table
        print("[Task] Picking tomato2")
        tomato2_pos = positions.get('tomato2')
        if tomato2_pos is None:
            raise KeyError("tomato2 not found in positions")
        obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Early termination after pick tomato2. Reward:", reward)
            return

        # 9) Place tomato2 on plate
        print("[Task] Placing tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Early termination after place tomato2. Reward:", reward)
            return

        # Task complete
        print("[Task] Completed plan. Final reward:", reward, "Done flag:", done)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()