from env import setup_environment, shutdown_environment
import numpy as np
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos   = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        item1_pos         = np.array(positions['item1'])
        item2_pos         = np.array(positions['item2'])
        plate_pos         = np.array(positions['plate'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to align with drawer pull direction (90 degrees about Z)
        print("[Task] Step 1: rotate gripper to 90 degrees for drawer alignment.")
        # Quaternion for 90° about Z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
        target_quat = np.array([0.0, 0.0, np.sin(np.deg2rad(45.0)), np.cos(np.deg2rad(45.0))])
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        # [Frozen Code End]
        if done:
            print("[Task] Ending early after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to the drawer side position
        print("[Task] Step 2: move gripper to drawer side position.")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Ending early after move-to-side. Reward:", reward)
            return

        # Step 3: Move gripper to the drawer anchor (handle) position
        print("[Task] Step 3: move gripper to drawer anchor position.")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ending early after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pull the drawer open
        print("[Task] Step 4: pull drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ending early after pull. Reward:", reward)
            return

        # Step 5: Pick the first tomato (item1) from the table
        print("[Task] Step 5: pick first tomato at position", item1_pos)
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Ending early after pick(item1). Reward:", reward)
            return

        # Step 6: Place the first tomato onto the plate
        print("[Task] Step 6: place first tomato on plate at position", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ending early after place(item1). Reward:", reward)
            return

        # Step 7: Pick the second tomato (item2) from the table
        print("[Task] Step 7: pick second tomato at position", item2_pos)
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Ending early after pick(item2). Reward:", reward)
            return

        # Step 8: Place the second tomato onto the plate
        print("[Task] Step 8: place second tomato on plate at position", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ending early after place(item2). Reward:", reward)
            return

        print("[Task] Completed all steps successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()