import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['tomato1']
        tomato2_pos       = positions['tomato2']
        plate_pos         = positions['plate']

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero to ninety degrees around the Y axis
        print("[Task] Step 1: rotate to 90 deg")
        # Construct a quaternion representing a 90° rotation about Y
        target_quat = R.from_euler('y', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended prematurely after rotate.")
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Task] Step 2: move to bottom side position:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Ended prematurely after move to side.")
            return

        # Step 3: Move gripper to the anchor position of the bottom drawer
        print("[Task] Step 3: move to bottom anchor position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended prematurely after move to anchor.")
            return

        # Step 4: Pull the drawer open by moving 0.1m along +X
        print("[Task] Step 4: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended prematurely after pull.")
            return

        # Step 5: Pick up tomato1 from the table
        print("[Task] Step 5: pick tomato1 at:", tomato1_pos)
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Ended prematurely after pick tomato1.")
            return

        # Step 6: Place tomato1 on the plate
        print("[Task] Step 6: place tomato1 on:", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended prematurely after place tomato1.")
            return

        # Step 7: Pick up tomato2 from the table
        print("[Task] Step 7: pick tomato2 at:", tomato2_pos)
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Ended prematurely after pick tomato2.")
            return

        # Step 8: Place tomato2 on the plate
        print("[Task] Step 8: place tomato2 on:", plate_pos)
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended prematurely after place tomato2.")
            return

        # Final check
        if done or reward >= 1.0:
            print("[Task] Goal achieved! All tomatoes placed.")
        else:
            print(f"[Task] Execution finished but goal not detected (done={done}, reward={reward})")

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()