import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, rotate, pull, pick, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        try:
            side_pos_bottom = np.array(positions['bottom_side_pos'])
            anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])
            tomato1_pos = np.array(positions['tomato1'])
            tomato2_pos = np.array(positions['tomato2'])
            plate_pos = np.array(positions['plate'])
        except KeyError as e:
            print(f"[Error] Missing object position for key: {e}")
            return

        # Compute the orientation quaternion for ninety_deg rotation
        target_quat = np.array([0.0, 0.0, np.sin(np.deg2rad(45.0)), np.cos(np.deg2rad(45.0))])
        # Compute pull distance between side and anchor positions
        pull_distance = float(np.linalg.norm(anchor_pos_bottom - side_pos_bottom))

        # === Execute Oracle Plan ===

        # Step 1: Move gripper to bottom side position
        print("[Task] Step 1: Moving to bottom side position:", side_pos_bottom)
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        if done:
            print("[Task] Early termination after Step 1. Reward:", reward)
            return

        # Step 2: Rotate gripper to ninety_deg
        print("[Task] Step 2: Rotating to ninety_deg with target quat:", target_quat)
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Early termination after Step 2. Reward:", reward)
            return

        # Step 3: Move gripper to bottom anchor position
        print("[Task] Step 3: Moving to bottom anchor position:", anchor_pos_bottom)
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Task] Early termination after Step 3. Reward:", reward)
            return

        # Step 4: Pull the drawer (distance = distance between anchor and side)
        print(f"[Task] Step 4: Pulling drawer by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance)
        if done:
            print("[Task] Early termination after Step 4. Reward:", reward)
            return

        # Step 5: Pick tomato1 from table
        print("[Task] Step 5: Picking tomato1 at:", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Early termination after Step 5. Reward:", reward)
            return

        # Step 6: Place tomato1 on plate
        print("[Task] Step 6: Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Early termination after Step 6. Reward:", reward)
            return

        # Step 7: Pick tomato2 from table
        print("[Task] Step 7: Picking tomato2 at:", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Early termination after Step 7. Reward:", reward)
            return

        # Step 8: Place tomato2 on plate
        print("[Task] Step 8: Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Early termination after Step 8. Reward:", reward)
            return

        # Final status
        if not done:
            print("[Task] Completed all steps. Final reward:", reward)
        else:
            print("[Task] Task ended with done=True. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()