# run_skeleton_task.py (Fully Generic Skeleton)
#
# Description:
#  - This code is a skeleton that shows how to set up an environment, retrieve
#    object positions, and execute a plan using predefined skills.
#  - It executes an oracle plan: pick item1 and item2 from the table and place them on the plate.
#  - It uses the provided pick and place skill functions without redefining them.

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Expecting keys: 'item1', 'item2', 'plate'
        positions = get_object_positions()
        try:
            item1_pos = positions['item1']
            item2_pos = positions['item2']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Missing object position for {e}. Available keys: {list(positions.keys())}")
            return

        # === Oracle Plan Execution ===

        # Step 1: pick item1 from table
        print("[Task] Step 1: Picking up item1 at:", item1_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick(item1).")
                return
        except Exception as e:
            print("[Error] pick(item1) failed:", e)
            return

        # Step 2: place item1 on plate
        print("[Task] Step 2: Placing item1 on plate at:", plate_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Task] Task ended during place(item1). Reward: {reward}")
                return
        except Exception as e:
            print("[Error] place(item1) failed:", e)
            return

        # Step 3: pick item2 from table
        print("[Task] Step 3: Picking up item2 at:", item2_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick(item2).")
                return
        except Exception as e:
            print("[Error] pick(item2) failed:", e)
            return

        # Step 4: place item2 on plate
        print("[Task] Step 4: Placing item2 on plate at:", plate_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Task] Task ended during place(item2). Reward: {reward}")
                return
        except Exception as e:
            print("[Error] place(item2) failed:", e)
            return

        # Final status
        print("[Task] Completed all plan steps. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()