import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys: 'tomato1', 'tomato2', 'plate', etc.
        tomato1_pos = positions.get('tomato1')
        tomato2_pos = positions.get('tomato2')
        plate_pos   = positions.get('plate')
        table_pos   = positions.get('table')  # if needed

        # Check that we have all required positions
        if tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise ValueError("Missing object positions for tomatoes or plate.")

        # === Execute Oracle Plan ===
        # Step 1: pick tomato1 from table
        print("[Task] Step 1: Picking tomato1 from table at", tomato1_pos)
        obs, reward, done = pick(
            env, task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after picking tomato1. Reward:", reward)
            return

        # Step 2: place tomato1 on plate
        print("[Task] Step 2: Placing tomato1 on plate at", plate_pos)
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after placing tomato1. Reward:", reward)
            return

        # Step 3: pick tomato2 from table
        print("[Task] Step 3: Picking tomato2 from table at", tomato2_pos)
        obs, reward, done = pick(
            env, task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after picking tomato2. Reward:", reward)
            return

        # Step 4: place tomato2 on plate
        print("[Task] Step 4: Placing tomato2 on plate at", plate_pos)
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after placing tomato2. Reward:", reward)
            return

        # Final check
        print("[Task] Completed all steps. Final reward:", reward, "Done flag:", done)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()