import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        try:
            positions = get_object_positions()
            tomato1_pos = positions['tomato1']
            plate_pos   = positions['plate']
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            return

        # === Task Plan Execution ===
        # Step 1: Pick tomato1 from the table
        print("[Task] Picking tomato1 at:", tomato1_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after pick. Reward:", reward)
            return

        # Step 2: Place tomato1 on the plate
        print("[Task] Placing tomato1 on plate at:", plate_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during place: {e}")
            return
        if done:
            print("[Task] Task completed successfully after place! Reward:", reward)
        else:
            print("[Task] Task step executed, done flag is False. Current reward:", reward)

        # Step 3: Pick tomato2 from the table
        try:
            tomato2_pos = positions['tomato2']
        except KeyError as e:
            print(f"[Error] Missing object tomato2 in positions: {e}")
            return
        print("[Task] Picking tomato2 at:", tomato2_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during pick tomato2: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after second pick. Reward:", reward)
            return

        # Step 4: Place tomato2 on the plate
        print("[Task] Placing tomato2 on plate at:", plate_pos)
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during place tomato2: {e}")
            return
        if done:
            print("[Task] Task completed successfully after placing second tomato! Reward:", reward)
        else:
            print("[Task] Task executed final step, done flag is False. Current reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()