import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys: 'tomato1', 'tomato2', 'plate'
        tomato1_pos = positions.get('tomato1')
        tomato2_pos = positions.get('tomato2')
        plate_pos   = positions.get('plate')
        if tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise KeyError("Missing one of: 'tomato1', 'tomato2', or 'plate' in object positions")

        # === Task Plan Execution ===
        # Step 1: Pick tomato1 from its initial location
        print("[Task] Step 1: Picking tomato1 at:", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick(tomato1). Reward:", reward)
            return

        # Step 2: Place tomato1 on the plate
        print("[Task] Step 2: Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during place(tomato1). Reward:", reward)
            return

        # Step 3: Pick tomato2 from its initial location
        print("[Task] Step 3: Picking tomato2 at:", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick(tomato2). Reward:", reward)
            return

        # Step 4: Place tomato2 on the plate
        print("[Task] Step 4: Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed after place(tomato2). Reward:", reward)
        else:
            print("[Task] Finished all steps. done flag:", done, "Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()