import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Choose the bottom drawer for opening
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper from zero_deg to ninety_deg about z-axis
        # Quaternion for 90° around z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
        target_quat = np.array([0.0, 0.0, 0.7071, 0.7071])
        print("[Plan] Step 1: rotate to ninety_deg orientation")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Task ended after rotate.")
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Plan] Step 2: move to bottom side position")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Task ended after move to side.")
            return

        # Step 3: Move gripper from side position to anchor (handle) position
        print("[Plan] Step 3: move to bottom anchor position")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after move to anchor.")
            return

        # Step 4: Pick the drawer handle at the anchor position
        print("[Plan] Step 4: pick bottom drawer handle")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after pick.")
            return

        # Step 5: Pull the drawer outward along the x-axis
        pull_distance = 0.1  # adjust as needed to open
        pull_axis = 'x'
        print(f"[Plan] Step 5: pull bottom drawer by {pull_distance}m along {pull_axis}-axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Plan] Task ended after pull.")
            return

        # Final check / summary
        print("[Plan] Completed all steps; bottom drawer should now be open. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()