import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from scipy.spatial.transform import Rotation as R

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()

        # Helper to map PDDL-style parameter names to position dictionary keys
        def map_param_to_key(param):
            parts = param.split('-')
            if len(parts) == 3:
                # e.g. ['side','pos','bottom'] -> 'bottom_side_pos'
                return f"{parts[2]}_{parts[0]}_{parts[1]}"
            return param

        # === Plan Execution ===

        # Step 1: Rotate gripper to ninety_deg
        print("[Task] Step 1: Rotate gripper to ninety_deg")
        angle_map = {
            'zero_deg': R.from_euler('z', 0, degrees=True).as_quat(),
            'ninety_deg': R.from_euler('z', 90, degrees=True).as_quat()
        }
        target_quat = angle_map['ninety_deg']

        # Prepare side and anchor positions for bottom drawer
        side_key = map_param_to_key('side-pos-bottom')
        if side_key not in positions:
            print(f"[Task] Position key '{side_key}' not found.")
            return
        side_pos = np.array(positions[side_key])

        anchor_key = map_param_to_key('anchor-pos-bottom')
        if anchor_key not in positions:
            print(f"[Task] Position key '{anchor_key}' not found.")
            return
        anchor_pos = np.array(positions[anchor_key])

        # === Frozen Code Start ===
        obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        # === Frozen Code End ===

        if done:
            print("[Task] Task ended early after moving to anchor!")
            return

        # Step 4: Pick at anchor-pos-bottom
        print("[Task] Step 4: Pick at anchor-pos-bottom")
        try:
            obs, reward, done = pick(env, task, target_pos=anchor_pos)
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Task] Error during pick: {e}")
            return

        # Step 5: Pull to open the drawer
        print("[Task] Step 5: Pull to open the drawer")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Error during pull: {e}")
            return

        print("[Task] Goal achieved: Drawer opened!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()