import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Identify an unlocked drawer ===
        # According to the initial state, 'middle' and 'top' are locked. Choose 'bottom'.
        drawer = 'bottom'
        side_key = f"{drawer}_side_pos"
        anchor_key = f"{drawer}_anchor_pos"
        joint_key = f"{drawer}_joint_pos"
        if side_key not in positions or anchor_key not in positions or joint_key not in positions:
            raise KeyError(f"Missing position entries for drawer '{drawer}' in positions dict")

        side_pos = np.array(positions[side_key])
        anchor_pos = np.array(positions[anchor_key])
        joint_pos = np.array(positions[joint_key])

        # Compute pull distance as the distance between anchor and joint positions
        pull_distance = float(np.linalg.norm(joint_pos - anchor_pos))

        # === Execute Oracle Plan Steps ===

        # Step 1: Rotate gripper to ninety degrees around z-axis
        print("[Task] Step 1: Rotating gripper to 90° around Z-axis")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw format

        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        # Frozen Code End

        if done:
            print("[Task] Task ended after rotation and initial move.")
            return

        # Step 3: Move gripper to the anchor position of the drawer
        print(f"[Task] Step 3: Moving to anchor position of '{drawer}' drawer:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended after moving to anchor position.")
            return

        # Step 4: Pick (grasp) the drawer at the anchor position
        print(f"[Task] Step 4: Picking up '{drawer}' drawer at anchor position")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Task ended after pick.")
            return

        # Step 5: Pull the drawer to open it
        print(f"[Task] Step 5: Pulling '{drawer}' drawer by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Pull completed but 'done' flag is False. Reward:", reward)

    except Exception as e:
        print("[Task] Error during execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()