import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Identify an accessible (unlocked) drawer: bottom (middle and top are locked)
        bottom_side_pos   = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to ninety degrees about the y-axis
        print("[Task] Step 1: Rotating gripper to 90 degrees")
        ninety_quat = R.from_euler('y', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Terminated after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Task] Step 2: Moving to side position of bottom drawer:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Terminated after move to side. Reward:", reward)
            return

        # Step 3: Move gripper to the anchor position of the bottom drawer
        print("[Task] Step 3: Moving to anchor position of bottom drawer:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Terminated after move to anchor. Reward:", reward)
            return

        # Step 4: Pick (grasp) the drawer handle at the anchor position
        print("[Task] Step 4: Grasping drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer to open it (unlock the cabinet)
        pull_distance = 0.1
        pull_axis = 'x'
        print(f"[Task] Step 5: Pulling drawer by {pull_distance}m along {pull_axis}-axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis=pull_axis,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task completed after pull! Reward:", reward)
        else:
            print("[Task] Pull step finished, done flag is False. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()