import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_bottom_pos = positions.get('bottom_side_pos')
        anchor_bottom_pos = positions.get('bottom_anchor_pos')
        if side_bottom_pos is None or anchor_bottom_pos is None:
            raise KeyError("Could not find required bottom positions in environment.")

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper from zero_deg to ninety_deg around Z-axis
        print("[Plan] Step 1: rotate to 90 degrees")
        theta = np.pi / 2.0
        # Quaternion [x, y, z, w] for 90° about Z
        target_quat = np.array([0.0, 0.0, np.sin(theta / 2.0), np.cos(theta / 2.0)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Finished early after rotate. Reward:", reward)
            return

        # Step 2: move gripper to the side position of the bottom drawer
        print("[Plan] Step 2: move to side-bottom position:", side_bottom_pos)
        obs, reward, done = move(env, task, side_bottom_pos)
        if done:
            print("[Plan] Finished early after move-to-side. Reward:", reward)
            return

        # Step 3: move gripper to the anchor position of the bottom drawer
        print("[Plan] Step 3: move to anchor-bottom position:", anchor_bottom_pos)
        obs, reward, done = move(env, task, anchor_bottom_pos)
        if done:
            print("[Plan] Finished early after move-to-anchor. Reward:", reward)
            return

        # Step 4: pick (grasp) the bottom drawer handle at the anchor position
        print("[Plan] Step 4: pick bottom drawer at anchor position")
        obs, reward, done = pick(env, task, anchor_bottom_pos)
        if done:
            print("[Plan] Finished early after pick. Reward:", reward)
            return

        # Step 5: pull the drawer fully open
        print("[Plan] Step 5: pull bottom drawer open")
        full_pull_distance = 0.2
        obs, reward, done = pull(env, task, full_pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task completed after pull. Reward:", reward)
        else:
            print("[Plan] Pull done, done flag is False. Reward:", reward)

    except Exception as e:
        print("[Error] Exception during task execution:", str(e))
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()