import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Execute Oracle Plan to Open a Drawer ===

        # 1) Rotate the gripper to 90 degrees around the z-axis
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Task] Rotating gripper to 90 degrees orientation.")
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended during rotate.")
            return

        # 2) Move to the side position of the bottom drawer
        bottom_side_pos = positions['bottom_side_pos']
        print(f"[Task] Moving to side position of bottom drawer: {bottom_side_pos}")
        obs, reward, done = move(env, task, np.array(bottom_side_pos))
        if done:
            print("[Task] Task ended during move-to-side.")
            return

        # 3) Move to the anchor position of the bottom drawer
        bottom_anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Moving to anchor position of bottom drawer: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Task] Task ended during move-to-anchor.")
            return

        # 4) Pick (grasp) the bottom drawer at the anchor position
        print("[Task] Picking the bottom drawer at anchor position.")
        obs, reward, done = pick(env, task, np.array(bottom_anchor_pos))
        if done:
            print("[Task] Task ended during pick.")
            return

        # 5) Pull the drawer to open it
        pull_distance = 0.2  # meters
        print(f"[Task] Pulling the drawer by {pull_distance}m along the x-axis.")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull.")
            return

        print("[Task] Completed plan execution successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
