from env import setup_environment, shutdown_environment
from skill_code import move, pick, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    """Execute the oracle plan to pull open any unlocked drawer."""
    print("===== Starting Drawer Open Task =====")
    env, task = setup_environment()
    try:
        # Reset the task and set up recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Plan Step 1: Rotate gripper to ninety degrees
        target_quat_for_ninety_deg = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print(f"[Plan] Step 1: rotate to ninety_deg quat {target_quat_for_ninety_deg}")

        # Plan Step 2: Move to bottom side position
        bottom_side_key = 'bottom_side_pos'
        if bottom_side_key not in positions:
            raise KeyError(f"Missing position for '{bottom_side_key}'")
        bottom_side_pos = np.array(positions[bottom_side_key])
        print(f"[Plan] Step 2: move to bottom side pos {bottom_side_pos}")

        # Plan Step 3: Move to bottom anchor position
        bottom_anchor_key = 'bottom_anchor_pos'
        if bottom_anchor_key not in positions:
            raise KeyError(f"Missing position for '{bottom_anchor_key}'")
        bottom_anchor_pos = np.array(positions[bottom_anchor_key])
        print(f"[Plan] Step 3: move to bottom anchor pos {bottom_anchor_pos}")

        obs, reward, done = rotate(env, task, target_quat_for_ninety_deg)
        obs, reward, done = move(env, task, bottom_side_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)

        if done:
            print("[Plan] Task ended during initial movements.")
            return

        # Plan Step 4: Pick the bottom drawer at anchor position
        print(f"[Plan] Step 4: pick drawer at {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended during pick.")
            return

        # Plan Step 5: Pull the drawer open
        pull_distance = 0.1
        print(f"[Plan] Step 5: pull drawer by {pull_distance}m along +x")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended during pull.")
        else:
            print("[Plan] Pull completed, drawer should be open.")

    finally:
        shutdown_environment(env)
        print("===== End of Drawer Open Task =====")

if __name__ == "__main__":
    run_skeleton_task()