import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define the keys for the bottom drawer positions
        side_key = 'bottom_side_pos'
        anchor_key = 'bottom_anchor_pos'
        if side_key not in positions:
            raise KeyError(f"Position key '{side_key}' not found in object positions.")
        if anchor_key not in positions:
            raise KeyError(f"Position key '{anchor_key}' not found in object positions.")
        side_pos = np.array(positions[side_key])
        anchor_pos = np.array(positions[anchor_key])

        # === Oracle Plan Execution ===

        # Step 1: rotate gripper to ninety_deg (90° around z-axis)
        print("[Plan] Step 1: rotate to ninety_deg")
        # Create quaternion for 90° around Z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Terminated after rotate. Reward:", reward)
            return

        # Step 2: move to bottom_side_pos
        print(f"[Plan] Step 2: move to {side_key} at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Terminated after move to side. Reward:", reward)
            return

        # Step 3: move to bottom_anchor_pos
        print(f"[Plan] Step 3: move to {anchor_key} at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated after move to anchor. Reward:", reward)
            return

        # Step 4: pick at bottom_anchor_pos
        print(f"[Plan] Step 4: pick at {anchor_key}")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Plan] Terminated after pick. Reward:", reward)
            return

        # Step 5: pull to open the drawer
        print("[Plan] Step 5: pull to open drawer (distance=0.1m along x-axis)")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Drawer opened successfully! Reward:", reward)
        else:
            print("[Plan] Pull action completed but task not marked done. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()