import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    from env import setup_environment, shutdown_environment
    from skill_code import move, pick, place, rotate, pull
    from video import init_video_writers, recording_step, recording_get_observation
    from object_positions import get_object_positions

    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys: 'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos', etc.

        # === Oracle Plan Execution ===

        # Step 1: Rotate the gripper to ninety degrees about the Z axis
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Terminated after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to the side position of bottom drawer
        print("[Plan] Step 2: move gripper to bottom_side_pos")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Terminated after move to side. Reward:", reward)
            return

        # Step 3: Move gripper to the anchor position of bottom drawer
        print("[Plan] Step 3: move gripper to bottom_anchor_pos")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Terminated after move to anchor. Reward:", reward)
            return

        # Step 4: Pick (grasp) the bottom drawer handle at anchor position
        print("[Plan] Step 4: pick bottom drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Terminated after pick. Reward:", reward)
            return

        # Step 5: Pull open the bottom drawer
        print("[Plan] Step 5: pull to open bottom drawer")
        joint_pos = np.array(positions['bottom_joint_pos'])
        pull_vector = joint_pos - anchor_pos
        pull_distance = float(np.linalg.norm(pull_vector))
        axis_index = int(np.argmax(np.abs(pull_vector)))
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        axis_name = axis_map[axis_index]
        if pull_vector[axis_index] < 0:
            pull_axis = '-' + axis_name
        else:
            pull_axis = axis_name
        obs, reward, done = pull(env, task, pull_distance, pull_axis=pull_axis)
        if done:
            print("[Plan] Drawer opened. Reward:", reward)
        else:
            print("[Plan] Completed pull; done flag is False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()