from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: Rotate the gripper to ninety degrees around the z-axis
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw format
        print("[Task] Step 1: Rotating gripper to ninety_deg orientation")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Ended after rotate. Reward:", reward)
            return

        # Step 2: Move to the side position of the bottom drawer
        side_bottom_pos = positions.get('bottom_side_pos')
        if side_bottom_pos is None:
            raise KeyError("Position for 'bottom_side_pos' not found in object positions.")
        print("[Task] Step 2: Moving to bottom drawer side position:", side_bottom_pos)
        obs, reward, done = move(env, task, side_bottom_pos)
        if done:
            print("[Task] Ended after move-to-side. Reward:", reward)
            return

        # Step 3: Move to the anchor position of the bottom drawer
        anchor_bottom_pos = positions.get('bottom_anchor_pos')
        if anchor_bottom_pos is None:
            raise KeyError("Position for 'bottom_anchor_pos' not found in object positions.")
        print("[Task] Step 3: Moving to bottom drawer anchor position:", anchor_bottom_pos)
        obs, reward, done = move(env, task, anchor_bottom_pos)
        if done:
            print("[Task] Ended after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick (grasp) the bottom drawer handle at the anchor position
        print("[Task] Step 4: Picking at anchor position of bottom drawer")
        obs, reward, done = pick(env, task, anchor_bottom_pos)
        if done:
            print("[Task] Ended after pick. Reward:", reward)
            return

        # Step 5: Pull to open the bottom drawer
        pull_distance = 0.2
        pull_axis = 'x'
        print(f"[Task] Step 5: Pulling drawer along {pull_axis}-axis by {pull_distance} meters")
        obs, reward, done = pull(env, task, pull_distance, pull_axis)
        if done:
            print("[Task] Goal achieved: Drawer opened! Reward:", reward)
        else:
            print("[Task] Drawer pull action completed. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
        raise
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()