import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Drawer handle side position, anchor position, and joint (opened) position
        bottom_side_pos   = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        bottom_joint_pos  = np.array(positions['bottom_joint_pos'])

        # === Oracle Plan for Opening Bottom Drawer ===
        # Step 1: Rotate gripper to ninety degrees
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Rotating gripper to ninety degrees orientation")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)

        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return

        # Step 2: Move to side position of bottom drawer
        print(f"[Task] Moving to side of bottom drawer at: {bottom_side_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)

        if done:
            print("[Task] Early termination after move-to-side. Reward:", reward)
            return

        # Step 3: Move to anchor position of bottom drawer
        print(f"[Task] Moving to anchor of bottom drawer at: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)

        if done:
            print("[Task] Early termination after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle at anchor position
        print(f"[Task] Picking drawer handle at: {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)

        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer to open (compute axis and distance)
        print("[Task] Computing pull direction and distance")
        delta = bottom_joint_pos - bottom_anchor_pos
        axis_idx = int(np.argmax(np.abs(delta)))
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        if delta[axis_idx] >= 0:
            pull_axis = axis_map[axis_idx]
        else:
            pull_axis = '-' + axis_map[axis_idx]
        pull_distance = float(abs(delta[axis_idx]))
        print(f"[Task] Pulling drawer along {pull_axis} by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis=pull_axis)

        if done:
            print("[Task] Drawer opened successfully. Reward:", reward)
        else:
            print("[Task] Drawer pull completed (done=False). Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()