import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # We will open the bottom drawer since top and middle are locked
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # Execute the oracle plan step by step
        print("===== Executing Plan: Open Bottom Drawer =====")

        # Step 1: Rotate the gripper (no change in orientation)
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]
        print("[Plan] Step 1: rotate to current orientation (no change).")
obs, reward, done = rotate(env, task, current_quat)
obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Task ended early during step 1 or 2.")
            return

        # Step 3: Move to the anchor (grasp) position of the bottom drawer
        print(f"[Plan] Step 3: move to bottom drawer anchor position: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended early during move-to-anchor.")
            return

        # Step 4: Pick (grasp) the bottom drawer handle at the anchor position
        print(f"[Plan] Step 4: pick at anchor position: {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Task ended early during pick.")
            return

        # Step 5: Pull the drawer open along its x-axis by a fixed distance
        pull_distance = 0.2  # meters to pull open
        pull_axis = 'x'
        print(f"[Plan] Step 5: pull drawer by {pull_distance}m along '{pull_axis}' axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Plan] Task ended early during pull.")
            return

        # Final check
        if reward is not None:
            print(f"[Plan] Completed open-drawer sequence. Final reward: {reward}")
        else:
            print("[Plan] Completed open-drawer sequence.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()