import numpy as np
from pyrep.objects.shape import Shape
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Run the oracle plan to select and fully open a drawer.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        side_pos = np.array(positions['bottom_side_pos'])
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        joint_pos = np.array(positions['bottom_joint_pos'])

        # Compute pull axis and distance from anchor to joint
        vec = joint_pos - anchor_pos
        idx = int(np.argmax(np.abs(vec)))
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        pull_axis = axis_map[idx]
        if vec[idx] < 0:
            pull_axis = '-' + pull_axis
        pull_distance = float(np.linalg.norm(vec))

        # Get target orientation from the side handle
        side_shape = Shape('bottom_side_pos')
        target_quat = np.array(side_shape.get_quaternion())

        print(f"[Plan] Step 1: rotate gripper to {target_quat}")
        print(f"[Plan] Step 2: move gripper to side position {side_pos}")
        print(f"[Plan] Step 3: move gripper to anchor position {anchor_pos}")
        print(f"[Plan] Step 4: pick drawer at anchor position")
        print(f"[Plan] Step 5: pull drawer by {pull_distance:.3f} along {pull_axis}")

        # === Oracle Plan Execution ===
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, side_pos)
obs, reward, done = move(env, task, anchor_pos)
obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.15, approach_axis='z')
obs, reward, done = pull(env, task, pull_distance, pull_axis=pull_axis)
        # [Frozen Code End]

        # Check result
        if done:
            print(f"[Result] Task finished successfully with reward: {reward}")
        else:
            print("[Result] Task completed plan steps but done=False")

    except Exception as e:
        print("[Error] Exception during plan execution:", e)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()