import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Select which drawer to open (bottom, middle or top)
        drawer = 'bottom'
        side_pos = positions[f"{drawer}_side_pos"]
        anchor_pos = positions[f"{drawer}_anchor_pos"]
        joint_pos = positions[f"{drawer}_joint_pos"]

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper from zero to ninety degrees around Y axis
        target_quat = R.from_euler('y', 90, degrees=True).as_quat()
        print(f"[Task] Rotating gripper to 90° around Y (quaternion: {target_quat})")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended after rotate. Reward:", reward)
            return

        # Step 2: Move to the side position of the drawer
        print(f"[Task] Moving to side position of {drawer} drawer: {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Ended after move-to-side. Reward:", reward)
            return

        # Step 3: Move to the anchor position of the drawer
        print(f"[Task] Moving to anchor position of {drawer} drawer: {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick (grasp) the drawer at its anchor point
        print(f"[Task] Picking up drawer at anchor: {anchor_pos}")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Ended after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer open along the axis defined by joint vs anchor
        diff = joint_pos - anchor_pos
        abs_diff = np.abs(diff)
        if abs_diff[0] >= abs_diff[1] and abs_diff[0] >= abs_diff[2]:
            axis = 'x' if diff[0] > 0 else '-x'
            distance = abs_diff[0]
        elif abs_diff[1] >= abs_diff[0] and abs_diff[1] >= abs_diff[2]:
            axis = 'y' if diff[1] > 0 else '-y'
            distance = abs_diff[1]
        else:
            axis = 'z' if diff[2] > 0 else '-z'
            distance = abs_diff[2]

        print(f"[Task] Pulling drawer along '{axis}' by {distance:.3f} meters")
        obs, reward, done = pull(env, task, distance, pull_axis=axis)
        if done:
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Pull action completed (done=False).")

    except Exception as e:
        print(f"[Task] Exception occurred during execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()