import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # === Task Plan ===
        # Goal: pull open any drawer that is not locked
        # We know 'middle' is locked; choose 'bottom' (also 'top' would work)
        drawer = 'bottom'
        side_key = f'{drawer}_side_pos'
        anchor_key = f'{drawer}_anchor_pos'
        if side_key not in positions or anchor_key not in positions:
            raise KeyError(f"Expected keys {side_key} and {anchor_key} in positions, got {list(positions.keys())}")
        side_pos = np.array(positions[side_key])
        anchor_pos = np.array(positions[anchor_key])

        # Step 1: Rotate gripper from zero to ninety degrees
        print(f"[Task] Step 1: rotate gripper to ninety degrees")
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        # Frozen Code End

        # Step 4: Pick at the anchor position to grasp the drawer handle
        print(f"[Task] Step 4: pick up drawer handle at {anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended during pick. Reward:", reward)
            return

        # Step 5: Pull the drawer to open it
        print(f"[Task] Step 5: pull to open drawer '{drawer}'")
        pull_distance = 0.1  # adjust if necessary
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Pull completed but 'done' is False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()