import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to ninety degrees
        print("[Plan] Step 1: rotate gripper to ninety degrees")
        # Quaternion for 90° about Z axis: [x, y, z, w]
        target_quat = np.array([0.0, 0.0, np.sqrt(2)/2, np.sqrt(2)/2])

        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        # Frozen Code End

        if done:
            print("[Plan] Task ended early with reward:", reward)
            return

        # Step 4: Pick (grasp) the bottom drawer at its anchor position
        print("[Plan] Step 4: pick bottom drawer at anchor position")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after pick with reward:", reward)
            return

        # Step 5: Pull the drawer open
        print("[Plan] Step 5: pull bottom drawer open")
        pull_distance = 0.1
        pull_axis = 'x'
        obs, reward, done = pull(env, task, pull_distance, pull_axis)
        if done:
            print("[Plan] Task ended after pull with reward:", reward)
            return

        # Plan finished without done=True
        print("[Plan] Completed all steps. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()