import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expecting keys: 'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos', etc.
        if 'bottom_side_pos' not in positions or 'bottom_anchor_pos' not in positions or 'bottom_joint_pos' not in positions:
            raise KeyError("Required bottom positions are missing in object positions.")

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper to side orientation.
        print("[Plan] Step 1: Rotating gripper to side orientation.")
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]
        obs, reward, done = rotate(env, task, target_quat=current_quat)
obs, reward, done = move(env, task, target_pos=side_pos)

        if done:
            print("[Plan] Task ended unexpectedly after rotate. Reward:", reward)
            return

        # Step 2: Move to drawer side position.
        side_pos = np.array(positions['bottom_side_pos'])
        print(f"[Plan] Step 2: Moving to bottom side position at {side_pos}.")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Task ended unexpectedly after move-to-side. Reward:", reward)
            return

        # Step 3: Move to drawer anchor position.
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        print(f"[Plan] Step 3: Moving to bottom anchor position at {anchor_pos}.")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle at anchor position.
        print(f"[Plan] Step 4: Picking drawer at anchor position {anchor_pos}.")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended unexpectedly after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer open.
        joint_pos = np.array(positions['bottom_joint_pos'])
        pull_distance = float(np.linalg.norm(joint_pos - anchor_pos))
        print(f"[Plan] Step 5: Pulling drawer with distance {pull_distance:.4f}.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended after pull. Reward:", reward)
        else:
            print("[Plan] Successfully executed all steps. Final reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()