import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Map PDDL names to position dictionary keys
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # Define quaternions for zero_deg and ninety_deg (about Z axis)
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # === Oracle Plan Execution ===
        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Plan] Terminated after rotate. Reward:", reward)
            return

        # Step 2: move to side-pos-bottom
        print("[Plan] Step 2: move to side-pos-bottom")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Plan] Terminated after move to side-pos-bottom. Reward:", reward)
            return

        # Step 3: move to anchor-pos-bottom
        print("[Plan] Step 3: move to anchor-pos-bottom")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after move to anchor-pos-bottom. Reward:", reward)
            return

        # Step 4: pick drawer at anchor-pos-bottom
        print("[Plan] Step 4: pick drawer at anchor-pos-bottom")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Plan] Terminated after pick. Reward:", reward)
            return

        # Step 5: pull drawer open
        print("[Plan] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Drawer opened successfully! Reward:", reward)
        else:
            print("[Plan] Finished pull but done=False. Reward:", reward)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()