import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Runs the sequence of skills to open the bottom drawer.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state and prepare recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos   = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        bottom_joint_pos  = np.array(positions['bottom_joint_pos'])

        # Step 1: Rotate the gripper from 0° to 90°
        from scipy.spatial.transform import Rotation as R
        rot = R.from_euler('z', 90, degrees=True)
        target_quat = rot.as_quat()       # [x, y, z, w]
        print(f"[Task] Rotating gripper to ninety degrees: {target_quat}")
        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        [Frozen Code End]
        if done:
            print("[Task] Terminated after rotate.")
            return

        # Step 2: Move to bottom side position
        print(f"[Task] Moving to bottom side position: {bottom_side_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Terminated after move to side position.")
            return

        # Step 3: Move to bottom anchor position
        print(f"[Task] Moving to bottom anchor position: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Terminated after move to anchor position.")
            return

        # Step 4: Grasp the drawer handle
        print(f"[Task] Grasping drawer handle at: {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Terminated after pick.")
            return

        # Step 5: Pull the drawer open
        pull_distance = np.linalg.norm(bottom_joint_pos - bottom_anchor_pos)
        print(f"[Task] Pulling drawer with distance = {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance=pull_distance)
        if done:
            print(f"[Task] Drawer opened successfully! Reward: {reward}")
        else:
            print("[Task] Pull action completed (done=False).")

    except Exception as e:
        print(f"[Task] Error during plan execution: {e}")
        raise
    finally:
        # Ensure environment is shut down even on error
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()