import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===
        print("[Plan] Step 1: rotate gripper to ninety degrees about Z")
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        
        # Steps 1-3 (frozen)
        obs, reward, done = rotate(env, task, ninety_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)

        if done:
            print(f"[Plan] Finished early after rotation/move steps. Reward: {reward}")
            return

        # Step 4: Pick the bottom drawer at the anchor position
        print("[Plan] Step 4: pick drawer at anchor position")
        try:
            obs, reward, done = pick(env, task, bottom_anchor_pos)
        except Exception as e:
            print(f"[Error] during pick at anchor-pos-bottom: {e}")
            raise
        if done:
            print(f"[Plan] Finished early after pick. Reward: {reward}")
            return

        # Step 5: Pull to open the drawer
        print("[Plan] Step 5: pull to open the drawer")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1)
        except Exception as e:
            print(f"[Error] during pull: {e}")
            raise
        if done:
            print(f"[Plan] Finished early after pull. Reward: {reward}")
            return

        print(f"[Plan] Completed all steps. Final reward: {reward}, done={done}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()